#!/bin/bash

# 🚀 DOCTORS HERO PROJECT STATUS COMMAND
# Run this command to give the AI assistant complete project context
# Usage: bash project-status.sh

echo "📋 ===== DOCTORS HERO PROJECT STATUS REPORT ====="
echo "🕐 Generated: $(date '+%Y-%m-%d %H:%M:%S %Z')"
echo ""

echo "🏗️ === PROJECT OVERVIEW ==="
echo "📁 Current Directory: $(pwd)"
echo "🌿 Git Branch: $(git branch --show-current)"
echo "📊 Git Status:"
git status --porcelain
echo ""

echo "📈 === RECENT ACTIVITY (Last 10 Commits) ==="
git log --oneline -10
echo ""

echo "🗂️ === COMPONENT STRUCTURE ==="
echo "📁 Layout Components:"
ls -la resources/js/components/layout/ 2>/dev/null || echo "❌ Layout folder not found"
echo ""
echo "📄 Page Components:"
ls -la resources/js/components/pages/ 2>/dev/null || echo "❌ Pages folder not found"
echo ""
echo "🔧 Common Components:"
ls -la resources/js/components/common/ 2>/dev/null || echo "❌ Common folder not found"
echo ""
echo "🔐 Auth Components:"
ls -la resources/js/components/auth/ 2>/dev/null || echo "❌ Auth folder not found"
echo ""
echo "🏥 Dashboard Components:"
ls -la resources/js/components/dashboard/ 2>/dev/null || echo "❌ Dashboard folder not found"
echo ""

echo "📝 === DOCUMENTATION STATUS ==="
echo "📋 Change Catalog:"
if [ -f "CHANGE_CATALOG.md" ]; then
    echo "✅ CHANGE_CATALOG.md exists ($(wc -l < CHANGE_CATALOG.md) lines)"
    echo "📅 Last entry date:"
    grep -o "### [0-9-]* [0-9:]* BD" CHANGE_CATALOG.md | tail -1
else
    echo "❌ CHANGE_CATALOG.md missing"
fi
echo ""

echo "🏛️ Component Architecture:"
if [ -f "COMPONENT_ARCHITECTURE.md" ]; then
    echo "✅ COMPONENT_ARCHITECTURE.md exists ($(wc -l < COMPONENT_ARCHITECTURE.md) lines)"
else
    echo "❌ COMPONENT_ARCHITECTURE.md missing"
fi
echo ""

echo "🎯 === CURRENT PROJECT STATE ==="
echo "🏠 Main Layout Status:"
if [ -f "resources/views/home.blade.php" ]; then
    echo "✅ home.blade.php exists (main website layout)"
else
    echo "❌ home.blade.php missing"
fi

if [ -f "resources/views/auth.blade.php" ]; then
    echo "✅ auth.blade.php exists (authentication layout)"
else
    echo "❌ auth.blade.php missing"
fi

if [ -f "resources/views/dashboard.blade.php" ]; then
    echo "✅ dashboard.blade.php exists (dashboard layout)"
else
    echo "❌ dashboard.blade.php missing"
fi

echo ""
echo "🏠 Homepage Status:"
if [ -f "resources/js/components/pages/HomePage.vue" ]; then
    echo "✅ HomePage.vue exists ($(wc -l < resources/js/components/pages/HomePage.vue) lines)"
else
    echo "❌ HomePage.vue missing"
fi

echo ""
echo "🚫 404 Page Status:"
if [ -f "resources/js/components/pages/NotFound.vue" ]; then
    echo "✅ NotFound.vue exists"
else
    echo "❌ NotFound.vue missing"
fi

echo ""
echo "🔐 Auth Pages Status:"
for page in LoginPage SignupPage; do
    if [ -f "resources/js/components/auth/${page}.vue" ]; then
        echo "✅ ${page}.vue exists ($(wc -l < resources/js/components/auth/${page}.vue) lines)"
    else
        echo "❌ ${page}.vue missing"
    fi
done

echo ""
echo "📱 Other Pages Status:"
for page in AboutPage ServicesPage ContactPage DoctorsPage AppointmentPage BlogPage; do
    if [ -f "resources/js/components/pages/${page}.vue" ]; then
        # Check if page redirects to 404
        if grep -q "router.push('/404')" "resources/js/components/pages/${page}.vue"; then
            echo "🔄 ${page}.vue → redirects to 404 (awaiting design)"
        else
            echo "✅ ${page}.vue → has content"
        fi
    else
        echo "❌ ${page}.vue missing"
    fi
done
echo ""

echo "🔧 === TECHNICAL STATUS ==="
echo "📦 Package.json:"
if [ -f "package.json" ]; then
    echo "✅ Package.json exists"
    echo "🔧 Build scripts available:"
    grep -A 5 '"scripts"' package.json | grep -E '"(dev|build|watch)"' | sed 's/^/  /'
else
    echo "❌ Package.json missing"
fi
echo ""

echo "🎨 CSS Status:"
if [ -f "public/css/style.css" ]; then
    echo "✅ Main CSS exists ($(wc -l < public/css/style.css) lines)"
    # Check smooth-wrapper width
    width=$(grep -A 1 "#smooth-wrapper" public/css/style.css | grep "width" | head -1 | sed 's/.*width: //' | sed 's/ !important.*//')
    echo "📐 #smooth-wrapper width: $width"
else
    echo "❌ Main CSS missing"
fi
echo ""

echo "🛠️ === DEVELOPMENT ENVIRONMENT ==="
echo "🟢 Node.js: $(node --version 2>/dev/null || echo '❌ Not installed')"
echo "📦 NPM: $(npm --version 2>/dev/null || echo '❌ Not installed')"
echo "🐘 PHP: $(php --version | head -1 2>/dev/null || echo '❌ Not installed')"
echo "🎼 Composer: $(composer --version 2>/dev/null | head -1 || echo '❌ Not installed')"
echo ""

echo "🚀 === QUICK BUILD TEST ==="
if command -v npm &> /dev/null; then
    echo "🔨 Running quick build check..."
    npm run build > /tmp/build_output.log 2>&1
    if [ $? -eq 0 ]; then
        echo "✅ Build successful"
        # Get build output size if available
        if [ -f "public/build/manifest.json" ]; then
            echo "📊 Assets compiled successfully"
        fi
    else
        echo "❌ Build failed - check logs"
    fi
else
    echo "⚠️ NPM not available - skipping build test"
fi
echo ""

echo "🎯 === AI ASSISTANT QUICK CONTEXT ==="
echo "🏥 PROJECT: Doctors Hero EMR SaaS Platform"
echo "⚙️ STACK: Laravel 12 + Vue.js 3 + Breeze + Alpine.js + Tailwind"
echo "🏗️ ARCHITECTURE: TRIPLE LAYOUT SYSTEM - Main + Auth + Dashboard"
echo "👤 DOCTOR LOGIN: doctor@doctorshero.com / password123"
echo "📋 WORKFLOW: Step-by-step, maintain CHANGE_CATALOG.md, no working ahead"
echo ""

echo "🔐 === COMPLETE AUTHENTICATION SYSTEM STATUS ==="
echo "✅ TRIPLE ARCHITECTURE IMPLEMENTED:"
echo "🏠 Main Website: home.blade.php → Vue.js SPA (resources/js/app.js)"
echo "🔑 Auth System: auth.blade.php → Vue.js Auth SPA (resources/js/auth.js)"
echo "🏥 Dashboard System: dashboard.blade.php → Vue.js Dashboard SPA (resources/js/dashboard.js)"
echo "📁 Dedicated Assets: public/auth/, public/dashboard/"
echo "🚫 NO PAGE RELOADS: All three systems are pure SPA with Vue Router"
echo ""

echo "🛣️ Complete Route Structure:"
echo "  🏠 / → home.blade.php (main website)"
echo "  🔐 /login → auth.blade.php (Vue LoginPage)"
echo "  📝 /signup → auth.blade.php (Vue SignupPage)"
echo "  🏥 /dashboard → redirects to role-based dashboard"
echo "  👨‍⚕️ /doctor-dashboard → dashboard.blade.php (Vue DoctorDashboard)"
echo "  👤 /patient-dashboard → dashboard.blade.php (Vue PatientDashboard)"
echo "  🚪 /logout → destroys session, redirects to login"
echo ""

echo "📡 API Endpoints (routes/api.php - CSRF-free):"
echo "  🔑 POST /api/login (dual auth: session + token)"
echo "  📝 POST /api/register (with validation)"
echo "  🚪 POST /api/logout (authenticated)"
echo "  👤 GET /api/user (authenticated)"
echo "  🤖 POST /api/ai-chat (AI chatbot)"
echo ""

echo "📚 === MEMORY TRIGGERS FOR AI ==="
echo "🔹 TRIPLE ARCHITECTURE: Main + Auth + Dashboard (all Vue.js SPAs)"
echo "🔹 Homepage is fully functional with complete design"
echo "🔹 Authentication system complete with professional template design"
echo "🔹 Dashboard system complete with doctor portal design (matching screenshot)"
echo "🔹 Other pages redirect to 404 until designs provided"
echo "🔹 Components organized: layout/, pages/, common/, auth/, dashboard/"
echo "🔹 AppHeader/AppFooter are reusable components"
echo "🔹 Vue Router handles all 404 scenarios in all systems"
echo "🔹 #smooth-wrapper CSS width issue was resolved (96% → 100%)"
echo "🔹 Laravel routes: Proper separation (web.php, api.php, auth.php)"
echo "🔹 Role-based authentication with session + token dual auth"
echo "🔹 Dashboard shows professional doctor interface with stats, calendar, appointments"
echo ""

echo "🤖 === AI CHATBOT STATUS ==="
echo "✅ D-Hero AI Assistant fully implemented and working"
echo "🔧 Backend: Laravel AiChatController with DeepSeek API integration"
echo "🎨 Frontend: AiChatBox.vue component with ChatGPT-style behavior"
echo "⚡ Animation: 0.1s transition delay for fast popup response"
echo "📍 Positioning: Initial search centered, chat mode moves to bottom (500%)"
echo "🛡️ Overlay Protection: Only close button (X) closes popup"
echo "🔑 API: DeepSeek v3.1 (sk-8d5c32bb4f054970aa5214ba311e4869)"
echo "📝 Documentation: All changes logged in CHANGE_CATALOG.md"
echo ""

echo "🎯 === AI CHAT TECHNICAL DETAILS ==="
if [ -f "resources/js/components/common/AiChatBox.vue" ]; then
    echo "✅ AiChatBox.vue exists ($(wc -l < resources/js/components/common/AiChatBox.vue) lines)"
else
    echo "❌ AiChatBox.vue missing"
fi

if [ -f "app/Http/Controllers/AiChatController.php" ]; then
    echo "✅ AiChatController.php exists ($(wc -l < app/Http/Controllers/AiChatController.php) lines)"
else
    echo "❌ AiChatController.php missing"
fi

if [ -f "config/ai.php" ]; then
    echo "✅ AI config file exists ($(wc -l < config/ai.php) lines)"
else
    echo "❌ AI config file missing"
fi

# Check CSS positioning fix
if grep -q "translate(-50%, 400%)" public/css/style.css 2>/dev/null; then
    echo "✅ CSS positioning fix applied (translate 400%)"
else
    echo "⚠️ CSS positioning may need verification"
fi
echo ""

echo "✅ === STATUS REPORT COMPLETE ==="
echo "🎯 Ready to continue development!"
echo "📝 Run this command anytime: bash project-status.sh"
echo ""